@extends('admin.layout')

@section('title', 'Edit SEO Page')

@section('content')
@php
    $pageKey = $seoPage->page_key;
    $pageLabel = $pageLabels[$pageKey] ?? \Illuminate\Support\Str::headline($pageKey);
    $keywords = collect(explode(',', (string) $seoPage->meta_keywords))
        ->map(fn ($keyword) => trim($keyword))
        ->filter()
        ->values();
    $selectedKeywords = old('meta_keywords', $keywords->all());
@endphp

<style>
.seo-edit-page {
    --seo-accent: #e6732c;
    --seo-border: #e4e8f1;
    --seo-text: #1f2937;
    --seo-muted: #6b7280;
}

.seo-edit-page .seo-head {
    padding: 1.1rem 1.35rem;
    border-radius: 16px;
    background: linear-gradient(140deg, #fff7ef 0%, #ffffff 65%, #fff4ea 100%);
    border: 1px solid #f3d4ba;
    box-shadow: 0 14px 28px rgba(15, 23, 42, 0.08);
}

.seo-edit-page .seo-kicker {
    margin: 0 0 .25rem 0;
    font-size: .74rem;
    text-transform: uppercase;
    letter-spacing: .11em;
    color: var(--seo-accent);
    font-weight: 700;
}

.seo-edit-page .seo-title {
    margin: 0;
    font-size: 1.7rem;
    color: var(--seo-text);
    font-weight: 700;
}

.seo-edit-page .form-card {
    border-radius: 18px;
    border: 1px solid var(--seo-border);
    background: #fff;
    box-shadow: 0 24px 52px rgba(15, 23, 42, 0.08);
    overflow: hidden;
}

.seo-edit-page .form-card-header {
    padding: 1rem 1.2rem;
    border-bottom: 1px solid #edf1f7;
    background: #fbfcff;
}

.seo-edit-page .field-label {
    font-size: .78rem;
    letter-spacing: .08em;
    text-transform: uppercase;
    color: #4b5a72;
    font-weight: 700;
    margin-bottom: .45rem;
}

.seo-edit-page .form-control,
.seo-edit-page .form-select {
    min-height: 46px;
    border-radius: 12px;
    border: 1px solid #d9deea;
}

.seo-edit-page .form-control:focus,
.seo-edit-page .form-select:focus {
    border-color: #ebb387;
    box-shadow: 0 0 0 .25rem rgba(230, 115, 44, 0.12);
}

.seo-edit-page .hint {
    font-size: .78rem;
    color: var(--seo-muted);
    margin-top: .35rem;
}

.seo-edit-page .primary-btn {
    border: none;
    border-radius: 12px;
    background: linear-gradient(135deg, #e6732c 0%, #f19b52 100%);
    box-shadow: 0 14px 26px rgba(230, 115, 44, 0.28);
    font-weight: 600;
}

.seo-edit-page .primary-btn:hover {
    background: linear-gradient(135deg, #ca5f1e 0%, #e6732c 100%);
}

.seo-edit-page .back-btn {
    border-radius: 12px;
    font-weight: 600;
}

.select2-container--default .select2-selection--multiple {
    min-height: 46px;
    border-radius: 12px;
    border: 1px solid #d9deea;
    padding: 6px 8px;
}

.select2-container--default.select2-container--focus .select2-selection--multiple {
    border-color: #ebb387;
    box-shadow: 0 0 0 .25rem rgba(230, 115, 44, 0.12);
}
</style>

<div class="seo-edit-page">
    <div class="seo-head mb-4 d-flex justify-content-between align-items-start flex-wrap gap-2">
        <div>
            <p class="seo-kicker">Admin Panel</p>
            <h2 class="seo-title">Edit SEO: {{ $pageLabel }}</h2>
        </div>
        <a href="{{ route('admin.seo-pages.index') }}" class="btn btn-outline-secondary back-btn">Back to SEO Pages</a>
    </div>

    @if(session('success'))
        <div class="alert alert-success">{{ session('success') }}</div>
    @endif

    @if ($errors->any())
        <div class="alert alert-danger">
            <ul class="mb-0">
                @foreach ($errors->all() as $error)
                    <li>{{ $error }}</li>
                @endforeach
            </ul>
        </div>
    @endif

    <div class="form-card">
        <div class="form-card-header">
            <strong style="color:#334155;">SEO Settings</strong>
        </div>

        <form action="{{ route('admin.seo-pages.update', $seoPage->page_key) }}" method="POST" class="p-4">
            @csrf
            @method('PUT')

            <div class="mb-3">
                <label class="field-label">Page Key</label>
                <input type="text" class="form-control" value="{{ $seoPage->page_key }}" readonly disabled>
            </div>

            <div class="mb-3">
                <label for="seo_title" class="field-label">SEO Title</label>
                <input
                    type="text"
                    id="seo_title"
                    name="seo_title"
                    value="{{ old('seo_title', $seoPage->seo_title) }}"
                    class="form-control @error('seo_title') is-invalid @enderror"
                    maxlength="60"
                    required
                >
                @error('seo_title')
                    <div class="invalid-feedback">{{ $message }}</div>
                @enderror
            </div>

            <div class="mb-3">
                <label for="seo_slug" class="field-label">SEO Slug</label>
                <input
                    type="text"
                    id="seo_slug"
                    name="seo_slug"
                    value="{{ old('seo_slug', $seoPage->seo_slug) }}"
                    class="form-control @error('seo_slug') is-invalid @enderror"
                    required
                >
                <div class="hint">Use path only, e.g. <code>/artworks</code></div>
                @error('seo_slug')
                    <div class="invalid-feedback">{{ $message }}</div>
                @enderror
            </div>

            <div class="mb-3">
                <label for="meta_description" class="field-label">Meta Description</label>
                <textarea
                    id="meta_description"
                    name="meta_description"
                    rows="4"
                    maxlength="160"
                    class="form-control @error('meta_description') is-invalid @enderror"
                >{{ old('meta_description', $seoPage->meta_description) }}</textarea>
                @error('meta_description')
                    <div class="invalid-feedback">{{ $message }}</div>
                @enderror
            </div>

            <div class="mb-4">
                <label for="meta_keywords" class="field-label">Meta Keywords</label>
                <select
                    id="meta_keywords"
                    name="meta_keywords[]"
                    class="form-select @error('meta_keywords') is-invalid @enderror"
                    multiple
                >
                    @foreach($selectedKeywords as $keyword)
                        <option value="{{ $keyword }}" selected>{{ $keyword }}</option>
                    @endforeach
                </select>
                <input
                    type="text"
                    id="meta_keyword_input"
                    class="form-control mt-2"
                    placeholder="Type keyword and press Enter or comma"
                >
                <div class="hint">Add keywords as tags. Press Enter or comma after each keyword.</div>
                @error('meta_keywords')
                    <div class="invalid-feedback">{{ $message }}</div>
                @enderror
                @error('meta_keywords.*')
                    <div class="invalid-feedback d-block">{{ $message }}</div>
                @enderror
            </div>

            <div class="d-flex gap-2 flex-wrap">
                <button type="submit" class="btn btn-primary primary-btn px-4">Save SEO</button>
                <a href="{{ route('admin.seo-pages.index') }}" class="btn btn-outline-secondary back-btn px-4">Cancel</a>
            </div>
        </form>
    </div>
</div>
@endsection

@push('styles')
<link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
@endpush

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function () {
    var $ = window.jQuery;
    var keywordSelect = document.getElementById('meta_keywords');
    var keywordInput = document.getElementById('meta_keyword_input');
    var form = keywordSelect ? keywordSelect.closest('form') : null;

    function hasKeyword(keyword) {
        var normalized = keyword.trim().toLowerCase();
        return Array.from(keywordSelect.options).some(function (option) {
            return option.value.trim().toLowerCase() === normalized;
        });
    }

    function addKeyword(rawValue) {
        if (!keywordSelect) return;

        var value = (rawValue || '').trim();
        if (!value) return;

        if (!hasKeyword(value)) {
            var option = new Option(value, value, true, true);
            keywordSelect.add(option);
        } else {
            Array.from(keywordSelect.options).forEach(function (option) {
                if (option.value.trim().toLowerCase() === value.toLowerCase()) {
                    option.selected = true;
                }
            });
        }

        if ($ && $.fn && $.fn.select2) {
            $(keywordSelect).trigger('change');
        }
    }

    if (keywordInput) {
        keywordInput.addEventListener('keydown', function (event) {
            if (event.key === 'Enter' || event.key === ',') {
                event.preventDefault();
                addKeyword(keywordInput.value.replace(/,+$/, ''));
                keywordInput.value = '';
            }
        });

        keywordInput.addEventListener('blur', function () {
            if (keywordInput.value.trim() !== '') {
                addKeyword(keywordInput.value);
                keywordInput.value = '';
            }
        });
    }

    if (form) {
        form.addEventListener('submit', function () {
            if (keywordInput && keywordInput.value.trim() !== '') {
                addKeyword(keywordInput.value);
                keywordInput.value = '';
            }
        });
    }

    if ($ && $.fn && $.fn.select2) {
        $(keywordSelect).select2({
            tags: true,
            tokenSeparators: [','],
            width: '100%',
            placeholder: 'Add keywords',
            allowClear: true,
            createTag: function (params) {
                var term = (params.term || '').trim();
                if (term === '') return null;
                return {
                    id: term,
                    text: term,
                    newTag: true
                };
            }
        });
    }
});
</script>
@endpush
