@extends('admin.layout')

@section('title', 'Add New Artwork')

@section('content')

<div class="painting-admin-page">
    <div class="paint-page-head mb-4">
        <p class="paint-kicker mb-1">Artwork Management</p>
        <h2 class="paint-title mb-1">Add New Artwork</h2>
        <p class="paint-subtitle mb-0">Build a clean, complete listing with premium presentation details.</p>
    </div>

{{-- ✅ VALIDATION ERRORS --}}
@if ($errors->any())
    <div class="alert alert-danger mb-3">
        <ul class="mb-0">
            @foreach ($errors->all() as $error)
                <li>{{ $error }}</li>
            @endforeach
        </ul>
    </div>
@endif

<form action="{{ route('admin.paintings.store') }}"
      method="POST"
      enctype="multipart/form-data"
      class="card shadow-sm p-4 painting-form-card border-0">

    @csrf

    {{-- 🔒 INTERNAL (OPTIONAL) --}}
    <input type="hidden" name="artwork_type_id" value="">

    {{-- ARTWORK CODE --}}
    <div class="mb-3">
        <label class="fw-semibold">Artwork Code</label>
        <input type="text"
               name="painting_code"
               value="{{ old('painting_code') }}"
               placeholder="Eg: ART-2026-001"
               class="form-control">
    </div>

    {{-- TITLE --}}
    <div class="mb-3">
        <label class="fw-semibold">Artwork Title <span class="text-danger">*</span></label>
        <input type="text"
               name="title"
               value="{{ old('title') }}"
               required
               class="form-control"
               placeholder="Enter artwork title">
    </div>

    {{-- UPDATED BY --}}
    <div class="mb-3">
        <label class="fw-semibold">Updated By</label>
        <input type="text"
               name="updated_by"
               value="{{ old('updated_by') }}"
               placeholder="Admin / Curator Name"
               class="form-control">
    </div>

    {{-- ARTIST --}}
    <div class="mb-3">
        <label class="fw-semibold">Artist</label>
        <select name="artist_id" class="form-select">
            <option value="">Select Artist</option>
            @foreach($artists as $id => $name)
                <option value="{{ $id }}" {{ old('artist_id') == $id ? 'selected' : '' }}>
                    {{ $name }}
                </option>
            @endforeach
        </select>
    </div>

     {{-- FORM --}}
<div class="mb-3">
    <label class="fw-semibold">Form</label>
    <select name="form_id" id="formSelect" class="form-select">
        <option value="">Select Form</option>
        @foreach($forms as $id => $name)
            <option value="{{ $id }}" {{ old('form_id') == $id ? 'selected' : '' }}>
                {{ $name }}
            </option>
        @endforeach
    </select>
</div>


 {{-- MEDIUM --}}
<div class="mb-3">
    <label class="fw-semibold">Medium</label>
    <select name="medium_id" id="mediumSelect" class="form-select">
        <option value="">Select Medium</option>
        @foreach($media as $medium)
            <option value="{{ $medium->id }}"
                    data-form-id="{{ $medium->form_id }}"
                    {{ old('medium_id') == $medium->id ? 'selected' : '' }}>
                {{ $medium->name }}
            </option>
        @endforeach
    </select>
</div>

    {{-- THEMES --}}
    <div class="mb-2">
        <label class="fw-semibold">Themes</label>
        <select id="categoryDropdown" class="form-select">
            <option value="">Select Theme</option>
            @foreach($categories as $category)
                <option value="{{ $category->id }}">{{ $category->name }}</option>
            @endforeach
        </select>
    </div>

    <div id="selectedCategories" class="d-flex flex-wrap gap-2 mb-2"></div>
    <div id="hiddenCategoryInputs"></div>

    <small class="text-muted d-block mb-3">
        Multiple themes select करें, × से remove करें
    </small>

   


   


    {{-- DESCRIPTION --}}
    <div class="mb-3">
        <label class="fw-semibold">Description</label>
        <textarea name="description" class="summernote">{{ old('description') }}</textarea>
    </div>

   {{-- PRICE --}}
<div class="mb-3">
    <label class="fw-semibold">Price</label>
    <input type="number"
           step="0.01"
           name="price"
           id="priceInput"
           value="{{ old('price') }}"
           class="form-control">
</div>

{{-- PRICE ON REQUEST --}}
<div class="form-check mb-4">
    <input class="form-check-input"
           type="checkbox"
           name="price_on_request"
           id="priceOnRequest"
           value="1"
           {{ old('price_on_request') ? 'checked' : '' }}>

    <label class="form-check-label fw-semibold" for="priceOnRequest">
        Price on Request
    </label>
</div>



    {{-- ARTWORK SIZE --}}
<div class="row mb-3">
    <div class="col-md-6">
        <label class="fw-semibold">Height</label>
        <input type="number"
               step="0.01"
               name="height"
               value="{{ old('height') }}"
               class="form-control"
               placeholder="Height (e.g. 24)">
    </div>

    <div class="col-md-6">
        <label class="fw-semibold">Width</label>
        <input type="number"
               step="0.01"
               name="width"
               value="{{ old('width') }}"
               class="form-control"
               placeholder="Width (e.g. 36)">
    </div>
</div>

<small class="text-muted d-block mb-3 painting-form-note">
    Example: Height × Width (in inches or cm)
</small>


    {{-- STATUS --}}
    <input type="hidden" name="status" value="active">

    {{-- IMAGE --}}
    <div class="mb-2">
        <label class="fw-semibold">Artwork Image</label>
        <input type="file"
               name="image"
               id="imageInput"
               accept="image/jpeg,image/png"
               class="form-control">
    </div>

    <small class="text-muted file-rules">
        ✔ JPG / PNG only<br>
        ✔ Max size: 5 MB<br>
        ✔ Recommended size: 1200 × 1200 px
    </small>

    <img id="previewImg"
         class="preview-artwork-image"
         style="display:none;">

    {{-- SUBMIT --}}
    <div class="mt-4">
        <button type="submit" class="btn btn-primary px-4 painting-primary-btn">
            Save Artwork
        </button>
    </div>

</form>
</div>
@endsection

@push('styles')
<link href="https://cdnjs.cloudflare.com/ajax/libs/summernote/0.8.18/summernote-lite.min.css" rel="stylesheet">
<style>
.painting-admin-page {
    --paint-accent: #e6732c;
    --paint-accent-soft: #fff2e8;
    --paint-accent-dark: #c95e1f;
    --paint-text: #1b2433;
    --paint-muted: #6b7280;
    --paint-border: #d9deea;
}

.painting-admin-page .paint-page-head {
    padding: 1.2rem 1.4rem;
    border-radius: 18px;
    background: linear-gradient(145deg, #fff9f3 0%, #ffffff 52%, #fff4ea 100%);
    border: 1px solid #f4d7bf;
    box-shadow: 0 16px 34px rgba(16, 24, 40, 0.08);
}

.painting-admin-page .paint-kicker {
    letter-spacing: .11em;
    text-transform: uppercase;
    font-weight: 700;
    color: var(--paint-accent);
    font-size: .75rem;
}

.painting-admin-page .paint-title {
    color: var(--paint-text);
    font-size: 1.8rem;
    font-weight: 700;
}

.painting-admin-page .paint-subtitle {
    color: var(--paint-muted);
    font-size: .95rem;
}

.painting-admin-page .painting-form-card {
    position: relative;
    border-radius: 22px;
    background: linear-gradient(170deg, #ffffff 0%, #fff9f3 56%, #ffffff 100%);
    box-shadow: 0 28px 60px rgba(15, 23, 42, 0.12);
    overflow: hidden;
}

.painting-admin-page .painting-form-card::before {
    content: "";
    position: absolute;
    top: -160px;
    right: -80px;
    width: 340px;
    height: 340px;
    border-radius: 999px;
    background: radial-gradient(circle, rgba(230, 115, 44, 0.14) 0%, rgba(230, 115, 44, 0) 72%);
    pointer-events: none;
}

.painting-admin-page .painting-form-card > * {
    position: relative;
    z-index: 1;
}

.painting-admin-page .painting-form-card label.fw-semibold {
    display: block;
    margin-bottom: .45rem;
    font-size: .76rem;
    letter-spacing: .08em;
    text-transform: uppercase;
    color: #4b5a72;
}

.painting-admin-page .painting-form-card .form-control,
.painting-admin-page .painting-form-card .form-select {
    min-height: 46px;
    border-radius: 12px;
    border: 1px solid var(--paint-border);
    background: #ffffff;
    box-shadow: inset 0 1px 2px rgba(15, 23, 42, 0.05);
    transition: all .2s ease;
}

.painting-admin-page .painting-form-card .form-control[type="file"] {
    padding-top: .58rem;
    padding-bottom: .58rem;
}

.painting-admin-page .painting-form-card .form-control:focus,
.painting-admin-page .painting-form-card .form-select:focus {
    border-color: #ebb387;
    box-shadow: 0 0 0 .25rem rgba(230, 115, 44, 0.12);
}

.painting-admin-page .painting-form-card .input-group .btn {
    border-radius: 0 12px 12px 0;
    border-color: var(--paint-border);
}

.painting-admin-page .painting-form-card .form-check {
    border: 1px dashed #f2bf9d;
    background: var(--paint-accent-soft);
    border-radius: 12px;
    padding: .85rem 1rem .85rem 2.2rem;
}

.painting-admin-page .painting-form-card .form-check-input {
    margin-top: .28rem;
}

.painting-admin-page .painting-form-note {
    color: #7b8394 !important;
    font-size: .83rem;
}

.painting-admin-page #selectedCategories {
    min-height: 44px;
    border: 1px dashed #d6deed;
    border-radius: 12px;
    background: #fff;
    padding: .55rem .6rem;
}

.painting-admin-page #selectedCategories .badge {
    background: #fff3e8 !important;
    color: #8f4619 !important;
    border: 1px solid #f3c5a3;
    font-weight: 600;
}

.painting-admin-page .file-rules {
    display: inline-block;
    margin-top: .35rem;
    padding: .65rem .8rem;
    border-radius: 10px;
    background: #fff7ef;
    border: 1px solid #f1d4bf;
}

.painting-admin-page .preview-artwork-image {
    width: 170px;
    margin-top: .8rem;
    border-radius: 12px;
    border: 1px solid #f0d9c8;
    box-shadow: 0 12px 24px rgba(15, 23, 42, 0.14);
}

.painting-admin-page .painting-primary-btn {
    border: none;
    border-radius: 12px;
    background: linear-gradient(135deg, var(--paint-accent) 0%, #f19b52 100%);
    box-shadow: 0 14px 26px rgba(230, 115, 44, 0.28);
    font-weight: 600;
    letter-spacing: .02em;
}

.painting-admin-page .painting-primary-btn:hover {
    background: linear-gradient(135deg, var(--paint-accent-dark) 0%, var(--paint-accent) 100%);
    transform: translateY(-1px);
}

.painting-admin-page .note-editor.note-frame {
    border: 1px solid var(--paint-border);
    border-radius: 12px;
    overflow: hidden;
}

.painting-admin-page .note-toolbar {
    background: #fff8f2;
    border-bottom: 1px solid #f1dac8;
}

.painting-admin-page .note-editing-area .note-editable {
    background: #fff;
}

.painting-admin-page .select2-container--default .select2-selection--single {
    height: 46px;
    border-radius: 12px;
    border: 1px solid var(--paint-border);
}

.painting-admin-page .select2-container--default .select2-selection--single .select2-selection__rendered {
    line-height: 44px;
    padding-left: 12px;
}

.painting-admin-page .select2-container--default .select2-selection--single .select2-selection__arrow {
    height: 44px;
    right: 8px;
}

.painting-admin-page .alert-danger {
    border: none;
    border-radius: 14px;
    box-shadow: 0 12px 22px rgba(153, 27, 27, 0.12);
}

@media (max-width: 768px) {
    .painting-admin-page .painting-form-card {
        padding: 1.25rem !important;
    }

    .painting-admin-page .paint-title {
        font-size: 1.45rem;
    }
}
</style>
@endpush

@push('scripts')
<script src="https://cdnjs.cloudflare.com/ajax/libs/summernote/0.8.18/summernote-lite.min.js"></script>
<script>
const priceCheckbox = document.getElementById('priceOnRequest');
const priceInput = document.getElementById('priceInput');

function togglePriceField() {
    if (priceCheckbox.checked) {
        priceInput.value = '';
        priceInput.setAttribute('disabled', true);
    } else {
        priceInput.removeAttribute('disabled');
    }
}

// Initial state (edit / validation error case)
togglePriceField();

// On change
priceCheckbox.addEventListener('change', togglePriceField);
</script>

<script>
$(function(){
    $('.summernote').summernote({
        height: 250,
        placeholder: 'Enter artwork description...'
    });
});

// IMAGE PREVIEW
const imageInput = document.getElementById('imageInput');
const previewImg = document.getElementById('previewImg');

imageInput.addEventListener('change', function(){
    const file = this.files[0];
    if(!file) return;

    if(file.size > 10 * 1024 * 1024){
        alert('Image size must be less than 10 MB');
        this.value = '';
        return;
    }

    previewImg.src = URL.createObjectURL(file);
    previewImg.style.display = 'block';
});

// THEME CHIP LOGIC
const dropdown = document.getElementById('categoryDropdown');
const selectedBox = document.getElementById('selectedCategories');
const hiddenInputs = document.getElementById('hiddenCategoryInputs');

dropdown.addEventListener('change', function () {
    const id = this.value;
    const text = this.options[this.selectedIndex].text;
    if (!id || document.getElementById('cat-'+id)) return;

    const chip = document.createElement('span');
    chip.id = 'cat-'+id;
    chip.className = 'badge bg-secondary px-3 py-2';
    chip.style.cursor = 'pointer';
    chip.innerHTML = `${text} <strong>×</strong>`;

    const input = document.createElement('input');
    input.type = 'hidden';
    input.name = 'category_ids[]';
    input.value = id;

    chip.onclick = () => {
        chip.remove();
        input.remove();
    };

    selectedBox.appendChild(chip);
    hiddenInputs.appendChild(input);
    this.value = '';
});
</script>

<script>
const formSelect   = document.getElementById('formSelect');
const mediumSelect = document.getElementById('mediumSelect');

function filterMediumByForm() {
    const selectedForm = formSelect.value;
    let hasValidMedium = false;

    Array.from(mediumSelect.options).forEach(option => {
        if (!option.value) return;

        const mediumForm = option.getAttribute('data-form-id');

        if (!selectedForm || mediumForm === selectedForm) {
            option.hidden = false;
            if (option.selected) hasValidMedium = true;
        } else {
            option.hidden = true;
            option.selected = false;
        }
    });

    if (!hasValidMedium) {
        mediumSelect.value = '';
    }
}

// Run on load (edit / validation error)
filterMediumByForm();

// Run on change
formSelect.addEventListener('change', filterMediumByForm);
</script>

<script>
(function () {
    var $ = window.jQuery;

    function toDigits(value) {
        return (value || "").replace(/\D/g, "");
    }

    function formatNumberWithCommas(value) {
        if (!value) return "";
        return value.replace(/\B(?=(\d{3})+(?!\d))/g, ",");
    }

    function getSummernotePlainText($el) {
        var html = $el.summernote("code") || "";
        var div = document.createElement("div");
        div.innerHTML = html;
        return (div.textContent || div.innerText || "").replace(/\u00a0/g, " ");
    }

    function setSummernotePlainText($el, text) {
        var safe = $("<div>").text(text).html().replace(/\n/g, "<br>");
        $el.summernote("code", safe);
    }

    document.addEventListener("DOMContentLoaded", function () {
        var formEl = document.querySelector("form[action*='admin.paintings.store'], form[action*='/admin/paintings']");
        var paintingCode = document.getElementById("paintingCode") || document.querySelector('input[name="painting_code"]');
        if (paintingCode && !paintingCode.id) paintingCode.id = "paintingCode";

        var autoUrl = document.getElementById("autoUrl");
        var copyBtn = document.getElementById("copyAutoUrlBtn");

        if (paintingCode && !autoUrl) {
            var wrap = document.createElement("div");
            wrap.className = "mt-2";

            var group = document.createElement("div");
            group.className = "input-group";

            autoUrl = document.createElement("input");
            autoUrl.type = "text";
            autoUrl.id = "autoUrl";
            autoUrl.className = "form-control";
            autoUrl.readOnly = true;
            autoUrl.placeholder = "Auto URL";

            copyBtn = document.createElement("button");
            copyBtn.type = "button";
            copyBtn.id = "copyAutoUrlBtn";
            copyBtn.className = "btn btn-outline-secondary";
            copyBtn.textContent = "Copy";

            group.appendChild(autoUrl);
            group.appendChild(copyBtn);
            wrap.appendChild(group);

            var codeGroup = paintingCode.closest(".mb-3") || paintingCode.parentElement;
            if (codeGroup) codeGroup.appendChild(wrap);
        } else if (autoUrl && !copyBtn) {
            copyBtn = document.createElement("button");
            copyBtn.type = "button";
            copyBtn.id = "copyAutoUrlBtn";
            copyBtn.className = "btn btn-outline-secondary mt-2";
            copyBtn.textContent = "Copy";
            autoUrl.insertAdjacentElement("afterend", copyBtn);
        }

        function sanitizeCode(v) {
            return (v || "").toUpperCase().replace(/[^A-Z0-9]/g, "").slice(0, 10);
        }

        function updateAutoUrl() {
            if (!paintingCode || !autoUrl) return;
            var code = sanitizeCode(paintingCode.value);
            paintingCode.value = code;
            autoUrl.value = code ? ("https://www.creativityartgallery.org/artworks/" + code) : "";
        }

        if (paintingCode) {
            paintingCode.setAttribute("maxlength", "10");
            paintingCode.addEventListener("input", function () {
                this.value = sanitizeCode(this.value);
            });
            paintingCode.addEventListener("blur", updateAutoUrl);
            if (paintingCode.value) updateAutoUrl();
        }

        if (copyBtn && autoUrl) {
            copyBtn.addEventListener("click", function () {
                var text = autoUrl.value || "";
                if (!text) return;
                if (navigator.clipboard && navigator.clipboard.writeText) {
                    navigator.clipboard.writeText(text);
                } else {
                    autoUrl.select();
                    document.execCommand("copy");
                    window.getSelection().removeAllRanges();
                }
            });
        }

        var artistSelect = document.querySelector('select[name="artist_id"]');
        if ($ && artistSelect && $.fn && $.fn.select2) {
            $(artistSelect).select2({
                width: "100%",
                placeholder: "Select Artist",
                allowClear: true
            });
        }

        var formSelect = document.getElementById("formSelect");
        var mediumSelect = document.getElementById("mediumSelect");

        function applyMediumDependency(resetMedium) {
            if (!formSelect || !mediumSelect) return;
            var selectedForm = formSelect.value;

            mediumSelect.disabled = !selectedForm;

            Array.from(mediumSelect.options).forEach(function (opt) {
                if (!opt.value) {
                    opt.hidden = false;
                    return;
                }
                var formId = opt.getAttribute("data-form-id");
                var show = !!selectedForm && formId === selectedForm;
                opt.hidden = !show;
                if (!show && opt.selected) opt.selected = false;
            });

            if (resetMedium) {
                mediumSelect.value = "";
            }
        }

        if (formSelect && mediumSelect) {
            applyMediumDependency(false);
            formSelect.addEventListener("change", function () {
                applyMediumDependency(true);
            });
        }

        var sizeInputs = document.querySelectorAll('input[name="height"], input[name="width"]');
        sizeInputs.forEach(function (inp) {
            inp.type = "text";
            inp.setAttribute("inputmode", "numeric");
            inp.setAttribute("maxlength", "3");
            inp.addEventListener("input", function () {
                this.value = toDigits(this.value).slice(0, 3);
            });
        });

        if ($) {
            var $desc = $('textarea[name="description"]');
            if ($desc.length) {
                if ($desc.next(".note-editor").length) {
                    $desc.summernote("destroy");
                }

                var syncing = false;
                $desc.summernote({
                    height: 250,
                    placeholder: "Enter artwork description...",
                    callbacks: {
                        onKeydown: function (e) {
                            var key = e.which || e.keyCode;
                            var isCtrl = e.ctrlKey || e.metaKey;
                            var allowedControl = [8, 9, 13, 27, 35, 36, 37, 38, 39, 40, 46].indexOf(key) !== -1;
                            if (isCtrl || allowedControl) return;

                            var len = getSummernotePlainText($desc).length;
                            if (len >= 400) {
                                e.preventDefault();
                            }
                        },
                        onPaste: function (e) {
                            e.preventDefault();
                            var clipboard = (e.originalEvent || e).clipboardData || window.clipboardData;
                            var pasteText = clipboard ? (clipboard.getData("text") || "") : "";
                            var current = getSummernotePlainText($desc);
                            var allowed = Math.max(0, 400 - current.length);
                            if (!allowed) return;
                            var insert = pasteText.slice(0, allowed);
                            document.execCommand("insertText", false, insert);
                        },
                        onChange: function () {
                            if (syncing) return;
                            var text = getSummernotePlainText($desc);
                            if (text.length > 400) {
                                syncing = true;
                                setSummernotePlainText($desc, text.slice(0, 400));
                                syncing = false;
                            }
                        }
                    }
                });

                var initText = getSummernotePlainText($desc);
                if (initText.length > 400) {
                    setSummernotePlainText($desc, initText.slice(0, 400));
                }
            }
        }

        var priceInput = document.getElementById("priceInput");
        var priceOnRequest = document.getElementById("priceOnRequest");

        function normalizePriceInput() {
            if (!priceInput) return;
            var digits = toDigits(priceInput.value).slice(0, 10);
            priceInput.value = formatNumberWithCommas(digits);
        }

        function togglePriceField() {
            if (!priceInput || !priceOnRequest) return;
            if (priceOnRequest.checked) {
                priceInput.value = "";
                priceInput.setAttribute("disabled", "disabled");
            } else {
                priceInput.removeAttribute("disabled");
                normalizePriceInput();
            }
        }

        if (priceInput) {
            priceInput.type = "text";
            priceInput.setAttribute("inputmode", "numeric");
            priceInput.addEventListener("input", normalizePriceInput);
            normalizePriceInput();
        }

        if (priceOnRequest) {
            togglePriceField();
            priceOnRequest.addEventListener("change", togglePriceField);
        }

        if (formEl && priceInput) {
            formEl.addEventListener("submit", function () {
                if (priceInput.disabled) return;
                priceInput.value = toDigits(priceInput.value).slice(0, 10);
            });
        }

        var imageInput = document.getElementById("imageInput");
        if (imageInput) {
            var fileNameBox = document.getElementById("imageFileName");
            if (!fileNameBox) {
                fileNameBox = document.createElement("div");
                fileNameBox.id = "imageFileName";
                fileNameBox.className = "small text-muted mt-1";
                imageInput.insertAdjacentElement("afterend", fileNameBox);
            }

            function updateFileName() {
                var file = imageInput.files && imageInput.files[0];
                fileNameBox.textContent = file ? ("Selected file: " + file.name) : "";
            }

            imageInput.addEventListener("change", updateFileName);
            updateFileName();
        }
    });
})();
</script>


@endpush
