import { haveMarketingConsent, haveAnalyticsConsent, havePreferencesConsent, haveSaleOfDataConsent, GPCSignal, setTrackingConsent, currentVisitorConsent, doesMerchantSupportGranularConsent, getCCPAConsent, getRegion, getTrackingConsent, getTrackingConsentMetafield, shouldShowGDPRBanner, userCanBeTracked, consentId } from './consent-methods.mjs';
import { getCCPAConsentValue, analyticsProcessingAllowed as analyticsProcessingAllowed$1, preferencesProcessingAllowed as preferencesProcessingAllowed$1, saleOfDataRegion, shouldShowBanner } from './consent-storage.mjs';
import { ConsentValues } from './types/index.mjs';
import { withBugsnag } from './bugsnag.mjs';
import { Monorail } from './monorail.mjs';
import { apiVersion } from './api-version.mjs';

function marketingAllowed() {
  return haveMarketingConsent();
}
function analyticsProcessingAllowed() {
  return haveAnalyticsConsent();
}
function preferencesProcessingAllowed() {
  return havePreferencesConsent();
}
function saleOfDataAllowed() {
  return haveSaleOfDataConsent();
}
const getMetadata = () => {
  return {
    name: "@shopify/consent-tracking-api",
    version: apiVersion,
    description: "Shopify Consent Tracking API"
  };
};

const consentAPI = ({
  useBugsnagReporting,
  useInstrumentation
}) => {
  // Shopify audiences relies on CCPA opt out events to respect CCPA.
  // To make it work with GPC we sync up the CCPA consent to the GPC value
  // when present, which as a side effect will send those events.
  // though in reality, this approach is flawed because Shopify Audiences
  // uses the IdentityGraph to match shopify_y to a Customer and does
  // not recheck after the initial event is fired (which is before there's
  // a chance to do an association between visitor to Customer) so the event
  // will never cause an opt Out of Audiences
  if (getCCPAConsentValue() != ConsentValues.DECLINED && GPCSignal() === false) {
    setTrackingConsent({
      sale_of_data: false
    }, () => false);
  }

  const api = {
    analyticsProcessingAllowed: analyticsProcessingAllowed$1,
    currentVisitorConsent,
    doesMerchantSupportGranularConsent,
    firstPartyMarketingAllowed: marketingAllowed,
    getCCPAConsent,
    getRegion,
    getTrackingConsent,
    getTrackingConsentMetafield,
    marketingAllowed,
    preferencesProcessingAllowed: preferencesProcessingAllowed$1,
    saleOfDataAllowed,
    saleOfDataRegion,
    setTrackingConsent,
    shouldShowBanner,
    shouldShowGDPRBanner,
    thirdPartyMarketingAllowed: saleOfDataAllowed,
    userCanBeTracked,
    consentId,
    unstable: {},
    __metadata__: getMetadata()
  }; // enable or disable the monorail singleton:

  new Monorail(useInstrumentation);
  if (!useBugsnagReporting) return api;
  const excludeKeys = ["unstable"];

  for (const key in api) {
    if (api.hasOwnProperty(key)) {
      // @ts-ignore
      api[key] = excludeKeys.includes(key) ? api[key] : withBugsnag(api[key]);
    }
  }

  return api;
};

function shopifyConsentAPI(options = {
  useBugsnagReporting: false,
  useInstrumentation: false
}) {
  return consentAPI(options);
}

export { analyticsProcessingAllowed, getMetadata, marketingAllowed, preferencesProcessingAllowed, saleOfDataAllowed, shopifyConsentAPI };
