<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

use App\Models\Artist;
use App\Models\Gallery;
use App\Models\Category;
use App\Models\Medium;
use App\Models\Form;
use App\Models\ArtworkType;
use App\Models\Favourite;

class Painting extends Model
{
    use HasFactory;

    protected $table = 'paintings';

    /* =====================
     * MASS ASSIGNMENT
     * ===================== */
    protected $fillable = [
        'title',
        'painting_code',
        'slug',
        'artist_id',
        'gallery_id',
        'artwork_type_id',
        'medium_id',
        'form_id',
        'description',
        'price',
        'price_on_request',
        'height',
        'width',
        'status',
        'image',
        'updated_by',
    ];

    /* =====================
     * CASTS
     * ===================== */
    protected $casts = [
        'price'            => 'decimal:2',
        'price_on_request' => 'boolean',
        'height'           => 'decimal:2',
        'width'            => 'decimal:2',
    ];

    /* =====================
     * RELATIONSHIPS
     * ===================== */

    public function artist()
    {
        return $this->belongsTo(Artist::class);
    }

    public function gallery()
    {
        return $this->belongsTo(Gallery::class);
    }

    public function favourites()
    {
        return $this->hasMany(Favourite::class);
    }

    public function categories()
    {
        return $this->belongsToMany(Category::class);
    }

    public function medium()
    {
        return $this->belongsTo(Medium::class);
    }

    public function form()
    {
        return $this->belongsTo(Form::class);
    }

    public function artworkType()
    {
        return $this->belongsTo(ArtworkType::class, 'artwork_type_id');
    }

    /* =====================
     * SCOPES
     * ===================== */

    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    /* =====================
     * HELPERS
     * ===================== */

    /**
     * Check if selected medium belongs to selected form
     * (Option 1 consistency helper)
     */
    public function isMediumValidForForm(): bool
    {
        if (!$this->medium || !$this->form) {
            return false;
        }

        return $this->medium->form_id === $this->form_id;
    }

    public function getFormattedSizeAttribute()
    {
        if ($this->height && $this->width) {
            return $this->height . ' × ' . $this->width;
        }

        return null;
    }

    public function setPaintingCodeAttribute($value): void
    {
        $this->attributes['painting_code'] = $value ? strtoupper($value) : null;
    }
}
