<?php

namespace App\Imports;

use App\Models\Artist;
use App\Models\Category;
use App\Models\Form;
use App\Models\Medium;
use App\Models\Painting;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Maatwebsite\Excel\Concerns\Importable;
use Maatwebsite\Excel\Concerns\RemembersChunkOffset;
use Maatwebsite\Excel\Concerns\SkipsEmptyRows;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithChunkReading;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use RuntimeException;
use Throwable;

class ArtworkBulkImport implements ToCollection, WithHeadingRow, WithChunkReading, SkipsEmptyRows
{
    use Importable;
    use RemembersChunkOffset;

    protected string $adminName;

    protected int $totalRows = 0;

    protected int $createdCount = 0;

    protected int $updatedCount = 0;

    protected int $failedCount = 0;

    protected array $errors = [];

    public function __construct(?string $adminName = null)
    {
        $this->adminName = trim((string) $adminName) !== '' ? trim((string) $adminName) : 'Admin';
    }

    public function collection(Collection $rows): void
    {
        DB::transaction(function () use ($rows): void {
            foreach ($rows as $index => $row) {
                $rowData = $this->normalizeRow($row->toArray());

                if ($this->isEmptyRow($rowData)) {
                    continue;
                }

                $this->totalRows++;
                $rowNumber = $this->resolveRowNumber((int) $index);

                try {
                    $payload = $this->validateAndPrepareRow($rowData);
                    $this->upsertPainting($payload);
                } catch (Throwable $e) {
                    $this->failedCount++;
                    $this->errors[] = [
                        'row' => $rowNumber,
                        'title' => (string) ($rowData['title'] ?? ''),
                        'painting_code' => (string) ($rowData['painting_code'] ?? ''),
                        'artist_name' => (string) ($rowData['artist_name'] ?? ''),
                        'status' => (string) ($rowData['status'] ?? ''),
                        'reason' => $e->getMessage(),
                    ];
                }
            }
        });
    }

    public function chunkSize(): int
    {
        return 100;
    }

    public function getSummary(): array
    {
        return [
            'total_rows' => $this->totalRows,
            'created' => $this->createdCount,
            'updated' => $this->updatedCount,
            'failed' => $this->failedCount,
        ];
    }

    public function getErrors(): array
    {
        return $this->errors;
    }

    protected function normalizeRow(array $row): array
    {
        $normalized = [];

        foreach ($row as $key => $value) {
            $normalized[(string) $key] = is_string($value) ? trim($value) : $value;
        }

        return $normalized;
    }

    protected function isEmptyRow(array $row): bool
    {
        foreach ($row as $value) {
            if ($value !== null && trim((string) $value) !== '') {
                return false;
            }
        }

        return true;
    }

    protected function resolveRowNumber(int $index): int
    {
        return $this->getChunkOffset() + $index + 2;
    }

    protected function validateAndPrepareRow(array $row): array
    {
        $title = trim((string) ($row['title'] ?? ''));
        if ($title === '') {
            throw new RuntimeException('title is required.');
        }

        $paintingCode = strtoupper(trim((string) ($row['painting_code'] ?? '')));
        if ($paintingCode === '') {
            throw new RuntimeException('painting_code is required.');
        }

        if (!preg_match('/^[A-Z0-9]{1,10}$/', $paintingCode)) {
            throw new RuntimeException('painting_code must be alphanumeric and maximum 10 characters.');
        }

        $artistName = trim((string) ($row['artist_name'] ?? ''));
        if ($artistName === '') {
            throw new RuntimeException('artist_name is required.');
        }

        $artist = Artist::query()
            ->whereRaw('LOWER(name) = ?', [mb_strtolower($artistName)])
            ->first();

        if (!$artist) {
            throw new RuntimeException('artist_name not found: ' . $artistName);
        }

        $status = strtolower(trim((string) ($row['status'] ?? '')));
        if (!in_array($status, ['active', 'sold', 'inactive'], true)) {
            throw new RuntimeException('status must be active, sold, or inactive.');
        }

        $priceProvided = $this->hasNonEmptyValue($row, 'price');
        $priceOnRequestProvided = $this->hasNonEmptyValue($row, 'price_on_request');
        $heightProvided = $this->hasNonEmptyValue($row, 'height');
        $widthProvided = $this->hasNonEmptyValue($row, 'width');
        $formProvided = $this->hasNonEmptyValue($row, 'form');
        $mediumProvided = $this->hasNonEmptyValue($row, 'medium');
        $themesProvided = $this->hasNonEmptyValue($row, 'themes');
        $descriptionProvided = $this->hasNonEmptyValue($row, 'description');
        $updatedByProvided = $this->hasNonEmptyValue($row, 'updated_by');

        $form = null;
        if ($formProvided) {
            $formName = trim((string) $row['form']);

            $form = Form::query()
                ->whereRaw('LOWER(name) = ?', [mb_strtolower($formName)])
                ->first();

            if (!$form) {
                throw new RuntimeException('form not found: ' . $formName);
            }
        }

        $medium = null;
        if ($mediumProvided) {
            if (!$form) {
                throw new RuntimeException('medium requires a valid form in the same row.');
            }

            $mediumName = trim((string) $row['medium']);

            $medium = Medium::query()
                ->whereRaw('LOWER(name) = ?', [mb_strtolower($mediumName)])
                ->where('form_id', $form->id)
                ->first();

            if (!$medium) {
                throw new RuntimeException('medium not found for selected form: ' . $mediumName);
            }
        }

        $price = null;
        if ($priceProvided) {
            $priceRaw = str_replace([',', ' '], '', (string) $row['price']);
            if (!is_numeric($priceRaw)) {
                throw new RuntimeException('price must be numeric.');
            }

            $price = (float) $priceRaw;
            if ($price < 0) {
                throw new RuntimeException('price must be greater than or equal to 0.');
            }
        }

        $priceOnRequest = false;
        if ($priceOnRequestProvided) {
            $priceOnRequestRaw = trim((string) $row['price_on_request']);
            if (!in_array($priceOnRequestRaw, ['0', '1'], true)) {
                throw new RuntimeException('price_on_request must be 0 or 1.');
            }

            $priceOnRequest = $priceOnRequestRaw === '1';
        }

        if ($status === 'sold') {
            $price = null;
            $priceOnRequest = true;
        } elseif ($priceOnRequest) {
            $price = null;
        }

        $height = null;
        if ($heightProvided) {
            $heightRaw = str_replace([',', ' '], '', (string) $row['height']);
            if (!is_numeric($heightRaw)) {
                throw new RuntimeException('height must be numeric.');
            }

            $height = (float) $heightRaw;
            if ($height < 0) {
                throw new RuntimeException('height must be greater than or equal to 0.');
            }
        }

        $width = null;
        if ($widthProvided) {
            $widthRaw = str_replace([',', ' '], '', (string) $row['width']);
            if (!is_numeric($widthRaw)) {
                throw new RuntimeException('width must be numeric.');
            }

            $width = (float) $widthRaw;
            if ($width < 0) {
                throw new RuntimeException('width must be greater than or equal to 0.');
            }
        }

        $description = $descriptionProvided ? trim((string) $row['description']) : null;

        $updatedBy = $updatedByProvided
            ? trim((string) $row['updated_by'])
            : $this->adminName;

        if (mb_strlen($updatedBy) > 100) {
            throw new RuntimeException('updated_by cannot exceed 100 characters.');
        }

        $themeIds = [];
        if ($themesProvided) {
            $themeNames = array_filter(array_map('trim', explode(',', (string) $row['themes'])));

            foreach ($themeNames as $themeName) {
                $themeIds[] = $this->findOrCreateCategoryId($themeName);
            }

            $themeIds = array_values(array_unique($themeIds));
        }

        return [
            'title' => $title,
            'painting_code' => $paintingCode,
            'artist_id' => $artist->id,
            'status' => $status,
            'price' => $price,
            'price_on_request' => $priceOnRequest,
            'height' => $height,
            'width' => $width,
            'form_id' => $form?->id,
            'medium_id' => $medium?->id,
            'description' => $description,
            'updated_by' => $updatedBy,
            'theme_ids' => $themeIds,
            'themes_provided' => $themesProvided,
            'price_provided' => $priceProvided,
            'price_on_request_provided' => $priceOnRequestProvided,
            'height_provided' => $heightProvided,
            'width_provided' => $widthProvided,
            'form_provided' => $formProvided,
            'medium_provided' => $mediumProvided,
            'description_provided' => $descriptionProvided,
        ];
    }

    protected function upsertPainting(array $payload): void
    {
        $painting = Painting::query()
            ->whereRaw('LOWER(painting_code) = ?', [mb_strtolower($payload['painting_code'])])
            ->first();

        $paintingExists = (bool) $painting;
        $fillable = (new Painting())->getFillable();

        $baseData = [
            'title' => $payload['title'],
            'painting_code' => $payload['painting_code'],
            'artist_id' => $payload['artist_id'],
            'status' => $payload['status'],
            'updated_by' => $payload['updated_by'],
        ];

        if (in_array('slug', $fillable, true)) {
            $baseData['slug'] = $payload['painting_code'];
        }

        if ($paintingExists) {
            $updateData = $baseData;

            if ($payload['status'] === 'sold') {
                $updateData['price'] = null;
                $updateData['price_on_request'] = true;
            } else {
                if ($payload['price_on_request_provided']) {
                    $updateData['price_on_request'] = $payload['price_on_request'];
                    if ($payload['price_on_request']) {
                        $updateData['price'] = null;
                    } elseif ($payload['price_provided']) {
                        $updateData['price'] = $payload['price'];
                    }
                } elseif ($payload['price_provided']) {
                    $updateData['price'] = $payload['price'];
                    $updateData['price_on_request'] = false;
                }
            }

            if ($payload['height_provided']) {
                $updateData['height'] = $payload['height'];
            }

            if ($payload['width_provided']) {
                $updateData['width'] = $payload['width'];
            }

            if ($payload['form_provided']) {
                $updateData['form_id'] = $payload['form_id'];
            }

            if ($payload['medium_provided']) {
                $updateData['medium_id'] = $payload['medium_id'];
            }

            if ($payload['description_provided']) {
                $updateData['description'] = $payload['description'];
            }

            $painting->update($updateData);
            $this->updatedCount++;
        } else {
            $createData = $baseData + [
                'price' => $payload['status'] === 'sold' ? null : $payload['price'],
                'price_on_request' => $payload['status'] === 'sold' ? true : $payload['price_on_request'],
                'height' => $payload['height'],
                'width' => $payload['width'],
                'form_id' => $payload['form_id'],
                'medium_id' => $payload['medium_id'],
                'description' => $payload['description'],
            ];

            $painting = Painting::query()->create($createData);
            $this->createdCount++;
        }

        if ($payload['themes_provided']) {
            $painting->categories()->sync($payload['theme_ids']);
        }
    }

    protected function findOrCreateCategoryId(string $name): int
    {
        $cleanName = trim($name);
        if ($cleanName === '') {
            throw new RuntimeException('themes contains an empty value.');
        }

        $existing = Category::query()
            ->whereRaw('LOWER(name) = ?', [mb_strtolower($cleanName)])
            ->first();

        if ($existing) {
            return $existing->id;
        }

        $baseSlug = Str::slug($cleanName);
        if ($baseSlug === '') {
            $baseSlug = 'theme';
        }

        $slug = $baseSlug;
        $counter = 1;

        while (Category::query()->where('slug', $slug)->exists()) {
            $slug = $baseSlug . '-' . $counter;
            $counter++;
        }

        $category = Category::query()->create([
            'name' => $cleanName,
            'slug' => $slug,
            'status' => 'active',
        ]);

        return $category->id;
    }

    protected function hasNonEmptyValue(array $row, string $key): bool
    {
        if (!array_key_exists($key, $row)) {
            return false;
        }

        $value = $row[$key];

        return $value !== null && trim((string) $value) !== '';
    }
}
