<?php

namespace App\Imports;

use App\Models\Artist;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Str;
use Maatwebsite\Excel\Concerns\Importable;
use Maatwebsite\Excel\Concerns\RemembersChunkOffset;
use Maatwebsite\Excel\Concerns\SkipsEmptyRows;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithChunkReading;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use RuntimeException;
use Throwable;

class ArtistBulkImport implements ToCollection, WithHeadingRow, WithChunkReading, SkipsEmptyRows
{
    use Importable;
    use RemembersChunkOffset;

    protected string $adminName;
    protected int $totalRows = 0;
    protected int $createdCount = 0;
    protected int $updatedCount = 0;
    protected int $failedCount = 0;
    protected array $errors = [];

    public function __construct(?string $adminName = null)
    {
        $this->adminName = trim((string) $adminName) !== '' ? trim((string) $adminName) : 'Admin';
    }

    public function collection(Collection $rows): void
    {
        DB::transaction(function () use ($rows): void {
            foreach ($rows as $index => $row) {
                $rowData = $this->normalizeRow($row->toArray());

                if ($this->isEmptyRow($rowData)) {
                    continue;
                }

                $this->totalRows++;
                $rowNumber = $this->resolveRowNumber((int) $index);

                try {
                    $payload = $this->validateAndPrepareRow($rowData);
                    $this->upsertArtist($payload);
                } catch (Throwable $e) {
                    $this->failedCount++;
                    $this->errors[] = [
                        'row' => $rowNumber,
                        'name' => (string) ($rowData['name'] ?? ''),
                        'status' => (string) ($rowData['status'] ?? ''),
                        'email' => (string) ($rowData['email'] ?? ''),
                        'username' => (string) ($rowData['username'] ?? ''),
                        'reason' => $e->getMessage(),
                    ];
                }
            }
        });
    }

    public function chunkSize(): int
    {
        return 100;
    }

    public function getSummary(): array
    {
        return [
            'total_rows' => $this->totalRows,
            'created' => $this->createdCount,
            'updated' => $this->updatedCount,
            'failed' => $this->failedCount,
        ];
    }

    public function getErrors(): array
    {
        return $this->errors;
    }

    protected function normalizeRow(array $row): array
    {
        $normalized = [];

        foreach ($row as $key => $value) {
            $normalized[(string) $key] = is_string($value) ? trim($value) : $value;
        }

        return $normalized;
    }

    protected function isEmptyRow(array $row): bool
    {
        foreach ($row as $value) {
            if ($value !== null && trim((string) $value) !== '') {
                return false;
            }
        }

        return true;
    }

    protected function resolveRowNumber(int $index): int
    {
        return $this->getChunkOffset() + $index + 2;
    }

    protected function validateAndPrepareRow(array $row): array
    {
        $name = trim((string) ($row['name'] ?? ''));
        if ($name === '') {
            throw new RuntimeException('name is required.');
        }

        $artist = Artist::query()
            ->whereRaw('LOWER(name) = ?', [mb_strtolower($name)])
            ->first();

        $statusInput = strtolower(trim((string) ($row['status'] ?? '')));
        if (!in_array($statusInput, ['active', 'inactive'], true)) {
            throw new RuntimeException('status must be active or inactive.');
        }

        // Existing DB enum supports active/blocked.
        $statusForDatabase = $statusInput === 'inactive' ? 'blocked' : 'active';

        $slugSource = $this->hasNonEmptyValue($row, 'slug')
            ? (string) $row['slug']
            : $name;
        $baseSlug = $this->sanitizeSlug($slugSource);
        if ($baseSlug === '') {
            $baseSlug = 'artist';
        }
        $slug = $this->makeUniqueSlug($baseSlug, $artist?->id);

        $emailProvided = $this->hasNonEmptyValue($row, 'email');
        $email = $emailProvided ? trim((string) $row['email']) : null;
        if ($emailProvided && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            throw new RuntimeException('email format is invalid.');
        }
        if ($emailProvided) {
            $emailExists = Artist::query()
                ->whereRaw('LOWER(email) = ?', [mb_strtolower($email)])
                ->when($artist, fn ($q) => $q->where('id', '!=', $artist->id))
                ->exists();

            if ($emailExists) {
                throw new RuntimeException('email already exists.');
            }
        }

        $usernameProvided = $this->hasNonEmptyValue($row, 'username');
        $username = $usernameProvided ? trim((string) $row['username']) : null;
        if ($usernameProvided) {
            $usernameExists = Artist::query()
                ->whereRaw('LOWER(username) = ?', [mb_strtolower($username)])
                ->when($artist, fn ($q) => $q->where('id', '!=', $artist->id))
                ->exists();

            if ($usernameExists) {
                throw new RuntimeException('username already exists.');
            }
        } elseif ($artist && !empty($artist->username)) {
            $username = $artist->username;
        } else {
            $username = $this->makeUniqueUsername($this->usernameBaseFromName($name), $artist?->id);
        }

        $isFeaturedProvided = $this->hasNonEmptyValue($row, 'is_featured');
        $isFeatured = $artist?->is_featured ?? false;
        if ($isFeaturedProvided) {
            $isFeaturedRaw = trim((string) $row['is_featured']);
            if (!in_array($isFeaturedRaw, ['0', '1'], true)) {
                throw new RuntimeException('is_featured must be 0 or 1.');
            }
            $isFeatured = $isFeaturedRaw === '1';
        }

        $phoneProvided = $this->hasNonEmptyValue($row, 'phone');
        $phone = $phoneProvided ? trim((string) $row['phone']) : null;

        $bioProvided = $this->hasNonEmptyValue($row, 'bio');
        $bio = $bioProvided ? trim((string) $row['bio']) : null;

        $updatedBy = $this->hasNonEmptyValue($row, 'updated_by')
            ? trim((string) $row['updated_by'])
            : $this->adminName;
        if (mb_strlen($updatedBy) > 100) {
            throw new RuntimeException('updated_by cannot exceed 100 characters.');
        }

        $profileImageFilename = null;
        $profileImageProvided = $this->hasNonEmptyValue($row, 'profile_image_url');
        if ($profileImageProvided) {
            $profileImageFilename = $this->downloadProfileImage((string) $row['profile_image_url']);
        }

        return [
            'artist' => $artist,
            'name' => $name,
            'slug' => $slug,
            'status' => $statusForDatabase,
            'email' => $email,
            'email_provided' => $emailProvided,
            'username' => $username,
            'phone' => $phone,
            'phone_provided' => $phoneProvided,
            'bio' => $bio,
            'bio_provided' => $bioProvided,
            'is_featured' => $isFeatured,
            'updated_by' => $updatedBy,
            'profile_image' => $profileImageFilename,
            'profile_image_provided' => $profileImageProvided,
        ];
    }

    protected function upsertArtist(array $payload): void
    {
        /** @var \App\Models\Artist|null $artist */
        $artist = $payload['artist'];

        if ($artist) {
            $updateData = [
                'name' => $payload['name'],
                'slug' => $payload['slug'],
                'status' => $payload['status'],
                'username' => $payload['username'],
                'is_featured' => $payload['is_featured'],
                'updated_by' => $payload['updated_by'],
            ];

            if ($payload['email_provided']) {
                $updateData['email'] = $payload['email'];
            }

            if ($payload['phone_provided']) {
                $updateData['phone'] = $payload['phone'];
            }

            if ($payload['bio_provided']) {
                $updateData['bio'] = $payload['bio'];
            }

            if ($payload['profile_image_provided']) {
                if (!empty($artist->profile_image)) {
                    $oldPath = public_path('uploads/artists/' . $artist->profile_image);
                    if (is_file($oldPath)) {
                        @unlink($oldPath);
                    }
                }
                $updateData['profile_image'] = $payload['profile_image'];
            }

            $artist->update($updateData);
            $this->updatedCount++;

            return;
        }

        Artist::query()->create([
            'name' => $payload['name'],
            'slug' => $payload['slug'],
            'email' => $payload['email'],
            'username' => $payload['username'],
            'phone' => $payload['phone'],
            'bio' => $payload['bio'],
            'status' => $payload['status'],
            'is_featured' => $payload['is_featured'],
            'updated_by' => $payload['updated_by'],
            'profile_image' => $payload['profile_image'],
            'password' => null,
        ]);

        $this->createdCount++;
    }

    protected function sanitizeSlug(string $value): string
    {
        return Str::of($value)->lower()->slug('-')->value();
    }

    protected function makeUniqueSlug(string $baseSlug, ?int $ignoreId = null): string
    {
        $slug = $baseSlug;
        $counter = 1;

        while (Artist::query()
            ->where('slug', $slug)
            ->when($ignoreId, fn ($q) => $q->where('id', '!=', $ignoreId))
            ->exists()) {
            $slug = $baseSlug . '-' . $counter;
            $counter++;
        }

        return $slug;
    }

    protected function usernameBaseFromName(string $name): string
    {
        $base = Str::of($name)
            ->lower()
            ->replaceMatches('/[^a-z0-9]+/', '')
            ->value();

        return $base !== '' ? Str::limit($base, 30, '') : 'artist';
    }

    protected function makeUniqueUsername(string $baseUsername, ?int $ignoreId = null): string
    {
        $username = $baseUsername;
        $counter = 1;

        while (Artist::query()
            ->whereRaw('LOWER(username) = ?', [mb_strtolower($username)])
            ->when($ignoreId, fn ($q) => $q->where('id', '!=', $ignoreId))
            ->exists()) {
            $suffix = (string) $counter;
            $trimmedBase = Str::limit($baseUsername, max(1, 30 - strlen($suffix)), '');
            $username = $trimmedBase . $suffix;
            $counter++;
        }

        return $username;
    }

    protected function downloadProfileImage(string $url): string
    {
        $cleanUrl = trim($url);
        if (!filter_var($cleanUrl, FILTER_VALIDATE_URL)) {
            throw new RuntimeException('profile_image_url is not a valid URL.');
        }

        $response = Http::timeout(30)->get($cleanUrl);
        if (!$response->successful()) {
            throw new RuntimeException('profile_image_url download failed.');
        }

        $mimeType = strtolower((string) $response->header('Content-Type'));
        if (!str_starts_with($mimeType, 'image/')) {
            throw new RuntimeException('profile_image_url does not point to an image.');
        }

        $extension = $this->resolveImageExtension($mimeType, $cleanUrl);
        $directory = public_path('uploads/artists');

        if (!is_dir($directory)) {
            File::makeDirectory($directory, 0755, true);
        }

        $filename = time() . '_bulk_profile_' . Str::random(8) . '.' . $extension;
        $path = $directory . DIRECTORY_SEPARATOR . $filename;

        if (@file_put_contents($path, $response->body()) === false) {
            throw new RuntimeException('profile image save failed.');
        }

        return $filename;
    }

    protected function resolveImageExtension(string $mimeType, string $url): string
    {
        $mime = trim(explode(';', $mimeType)[0]);
        $mimeMap = [
            'image/jpeg' => 'jpg',
            'image/jpg' => 'jpg',
            'image/png' => 'png',
            'image/gif' => 'gif',
            'image/webp' => 'webp',
            'image/bmp' => 'bmp',
            'image/svg+xml' => 'svg',
        ];

        if (isset($mimeMap[$mime])) {
            return $mimeMap[$mime];
        }

        $path = parse_url($url, PHP_URL_PATH) ?: '';
        $extension = strtolower(pathinfo($path, PATHINFO_EXTENSION));
        if ($extension !== '') {
            return $extension;
        }

        return 'jpg';
    }

    protected function hasNonEmptyValue(array $row, string $key): bool
    {
        if (!array_key_exists($key, $row)) {
            return false;
        }

        $value = $row[$key];

        return $value !== null && trim((string) $value) !== '';
    }
}
