<?php

namespace App\Http\Controllers\Front;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;

// MODELS
use App\Models\HeroSlide;
use App\Models\Artist;
use App\Models\Category;
use App\Models\Painting;
use App\Models\Event;
use App\Models\AboutSection;
use App\Models\AboutCreativitySection;
use App\Models\Exhibition;
use App\Models\ExhibitionImage;
use App\Models\ContactSetting;
use App\Models\ContactEnquiry;
use App\Models\ArtAdvisory;

// MAIL
use App\Mail\ContactEnquiryMail;

class FrontController extends Controller
{
    /* ======================
     * HOME PAGE
     * ====================== */
     public function index()
    {
        $heroSlides = HeroSlide::where('status', 'active')
            ->orderBy('sort_order')
            ->get();

        $featuredArtists = Artist::where('status', 'active')
            ->where('is_featured', 1)
            ->latest()
            ->take(10)
            ->get();

        $categories = Category::where('status', 'active')
            ->orderBy('name')
            ->get();

        $events = Event::latest()->take(2)->get();

        $aboutCreativity = AboutCreativitySection::where('status', 'active')->first();

        /* ✅ FEATURED EXHIBITIONS (FIXED – NO DATE BLOCKING) */
        $featuredExhibitions = Exhibition::where('is_featured', 1)
            ->latest()
            ->get();

        return view('front.home', compact(
            'heroSlides',
            'featuredArtists',
            'categories',
            'events',
            'aboutCreativity',
            'featuredExhibitions'
        ));
    }
    /* ======================
     * ABOUT PAGE
     * ====================== */
    public function about()
    {
        $about = AboutSection::first();
        return view('front.about', compact('about'));
    }

    /* ======================
     * ARTIST DETAIL
     * ====================== */
    public function artistDetail($slug)
    {
        $artist = Artist::where('slug', $slug)
            ->with('paintings')
            ->firstOrFail();

        return view('front.artist_detail', compact('artist'));
    }

    /* ======================
     * CATEGORY PAGE
     * ====================== */
    public function categoryPage($slug)
    {
        $category = Category::where('slug', $slug)->firstOrFail();

        $paintings = Painting::with('artist')
            ->whereHas('categories', function ($q) use ($category) {
                $q->where('categories.id', $category->id);
            })
            ->latest()
            ->get();

        return view('front.category_page', compact('category', 'paintings'));
    }

    /* ======================
     * EXHIBITIONS LIST
     * ====================== */
    public function exhibitions()
     {
         $exhibitions = Exhibition::latest()->get();
         return view('front.exhibitions.index', compact('exhibitions'));
     }

    /* ======================
     * EXHIBITION DETAIL
     * ====================== */
    public function exhibitionDetail($slug)
    {
        $exhibition = Exhibition::where('slug', $slug)
            ->with('images')
            ->firstOrFail();

        return view('front.exhibitions.show', compact('exhibition'));
    }

    /* ======================
     * CONTACT PAGE
     * ====================== */
    public function contact()
    {
        $contact = ContactSetting::first();
        return view('front.contact', compact('contact'));
    }

    /* ======================
     * CONTACT FORM SUBMIT
     * ====================== */
    public function contactSubmit(Request $request)
    {
        $data = $request->validate([
            'name'    => 'required|string|max:255',
            'email'   => 'required|email|max:255',
            'phone'   => 'nullable|string|max:20',
            'subject' => 'nullable|string|max:255',
            'message' => 'required|string',
        ]);

        // Save enquiry
        $enquiry = ContactEnquiry::create($data);

        // Send email to admin
        Mail::to(config('mail.from.address'))
            ->send(new ContactEnquiryMail($enquiry));

        return back()->with('success', 'Thank you! We will contact you soon.');
    }

    /* ======================
     * ART ADVISORY PAGE
     * ====================== */
    public function artAdvisory()
    {
        $advisory = ArtAdvisory::where('status', 1)->first();
        return view('front.art-advisory', compact('advisory'));
    }
}
