<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\PageContent;
use App\Models\SeoPage;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Str;

class PageContentController extends Controller
{
    public function edit($page)
    {
        $content = PageContent::firstOrCreate(['page_key' => $page]);
        $seoPageKey = $this->resolveSeoPageKey($page);

        $seoPage = SeoPage::query()->where('page_key', $seoPageKey)->first();

        if (!$seoPage) {
            $seoPage = new SeoPage(array_merge(
                ['page_key' => $seoPageKey],
                $this->defaultSeoValues($seoPageKey)
            ));
        }

        return view('admin.page_contents.edit', compact('content', 'seoPage', 'page'));
    }

    public function update(Request $request, $page)
    {
        $request->validate([
            'content' => 'nullable|string',
            'seo_title' => 'required|string|max:60',
            'seo_slug' => 'required|string|max:255',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:160',
            'meta_keywords' => 'nullable|string',
            'canonical_url' => 'nullable|string|max:2048',
            'meta_robots' => 'nullable|in:index,noindex,follow,nofollow',
            'og_title' => 'nullable|string|max:255',
            'og_description' => 'nullable|string',
            'og_image' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:5120',
            'og_type' => 'nullable|in:website,article',
            'og_url' => 'nullable|string|max:2048',
            'twitter_card' => 'nullable|in:summary,summary_large_image',
            'twitter_title' => 'nullable|string|max:255',
            'twitter_description' => 'nullable|string',
            'twitter_image' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:5120',
            'schema_markup' => 'nullable|string',
            'breadcrumb_title' => 'nullable|string|max:255',
        ]);

        PageContent::updateOrCreate(
            ['page_key' => $page],
            [
                'content' => $request->content,
                'status' => 1,
            ]
        );

        $seoPageKey = $this->resolveSeoPageKey($page);
        $existingSeoPage = SeoPage::query()->where('page_key', $seoPageKey)->first();
        $defaults = $this->defaultSeoValues($seoPageKey);
        $seoColumns = $this->seoPageColumns();
        $payload = [];

        $fieldMap = [
            'seo_title' => $request->seo_title ?: ($existingSeoPage->seo_title ?? $defaults['seo_title']),
            'seo_slug' => SeoPage::normalizeSlug((string) ($request->seo_slug ?: ($existingSeoPage->seo_slug ?? $defaults['seo_slug']))),
            'meta_title' => $request->meta_title,
            'meta_description' => $request->meta_description,
            'meta_keywords' => $request->meta_keywords,
            'canonical_url' => $request->canonical_url,
            'meta_robots' => $request->meta_robots,
            'og_title' => $request->og_title,
            'og_description' => $request->og_description,
            'og_type' => $request->og_type,
            'og_url' => $request->og_url,
            'twitter_card' => $request->twitter_card,
            'twitter_title' => $request->twitter_title,
            'twitter_description' => $request->twitter_description,
            'schema_markup' => $request->schema_markup,
            'breadcrumb_title' => $request->breadcrumb_title,
        ];

        foreach ($fieldMap as $column => $value) {
            if (in_array($column, $seoColumns, true)) {
                $payload[$column] = $value;
            }
        }

        if (in_array('og_image', $seoColumns, true)) {
            $uploadedOgImage = $this->storeSeoImage($request, 'og_image');
            if ($uploadedOgImage !== null) {
                $payload['og_image'] = $uploadedOgImage;
            }
        }

        if (in_array('twitter_image', $seoColumns, true)) {
            $uploadedTwitterImage = $this->storeSeoImage($request, 'twitter_image');
            if ($uploadedTwitterImage !== null) {
                $payload['twitter_image'] = $uploadedTwitterImage;
            }
        }

        SeoPage::updateOrCreate(['page_key' => $seoPageKey], $payload);
        Cache::forget('seo_page_slug_map');

        return back()->with('success', 'Page content updated successfully');
    }

    private function resolveSeoPageKey(string $page): string
    {
        $map = [
            'exhibitions' => 'exclusive',
            'blogs' => 'blog',
            'art-advisory' => 'art_advisory',
        ];

        return $map[$page] ?? $page;
    }

    private function defaultSeoValues(string $seoPageKey): array
    {
        $defaults = [
            'home' => ['seo_title' => 'Home', 'seo_slug' => ''],
            'about' => ['seo_title' => 'About', 'seo_slug' => 'about'],
            'exclusive' => ['seo_title' => 'Exclusive', 'seo_slug' => 'exhibitions'],
            'artworks' => ['seo_title' => 'Artworks', 'seo_slug' => 'artworks'],
            'artists' => ['seo_title' => 'Artists', 'seo_slug' => 'artists'],
            'blog' => ['seo_title' => 'Blog', 'seo_slug' => 'blogs'],
            'art_advisory' => ['seo_title' => 'Art Advisory', 'seo_slug' => 'art-advisory'],
            'contact' => ['seo_title' => 'Contact', 'seo_slug' => 'contact'],
        ];

        if (isset($defaults[$seoPageKey])) {
            return $defaults[$seoPageKey];
        }

        return [
            'seo_title' => Str::headline(str_replace('_', ' ', $seoPageKey)),
            'seo_slug' => Str::slug(str_replace('_', ' ', $seoPageKey)),
        ];
    }

    private function seoPageColumns(): array
    {
        static $columns = null;

        if ($columns === null) {
            $columns = Schema::hasTable('seo_pages') ? Schema::getColumnListing('seo_pages') : [];
        }

        return $columns;
    }

    private function storeSeoImage(Request $request, string $field): ?string
    {
        if (!$request->hasFile($field)) {
            return null;
        }

        $file = $request->file($field);
        if (!$file || !$file->isValid()) {
            return null;
        }

        $directory = public_path('uploads/seo_pages');
        if (!is_dir($directory)) {
            mkdir($directory, 0755, true);
        }

        $filename = $field . '_' . time() . '_' . Str::random(8) . '.' . $file->getClientOriginalExtension();
        $file->move($directory, $filename);

        return 'uploads/seo_pages/' . $filename;
    }
}
