<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use App\Models\Medium;
use App\Models\Form;
use Auth;

class MediumController extends Controller
{
    /**
     * LIST PAGE
     */
    public function index(Request $request)
    {
        $forms = Form::where('status', 'active')->get();
        $selectedFormId = $request->integer('form_id');

        // Mediums screen should load after selecting a form and clicking "Go".
        $mediums = collect();
        if ($selectedFormId > 0) {
            $mediums = Medium::with('form')
                ->withCount('paintings')
                ->where('form_id', $selectedFormId)
                ->orderBy('id', 'DESC')
                ->get();
        }

        return view('admin.mediums.index', compact('mediums', 'forms', 'selectedFormId'));
    }

    /**
     * STORE (CREATE)
     */
    public function store(Request $request)
    {
        $request->validate([
            'name'    => 'required|string|max:255',
            'form_id' => 'required|exists:forms,id',
            'status'  => 'nullable|in:active,inactive',
        ]);

        Medium::create([
            'name'       => $request->name,
            'slug'       => Str::slug($request->name),
            'form_id'    => $request->form_id,
            'status'     => $request->status ?? 'active',
            'updated_by' => Auth::guard('admin')->user()->name ?? 'admin',
        ]);

        return redirect()->back()->with('success', 'Medium added successfully');
    }

    /**
     * UPDATE
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'name'    => 'required|string|max:255',
            'form_id' => 'required|exists:forms,id',
            'status'  => 'nullable|in:active,inactive',
        ]);

        $medium = Medium::findOrFail($id);

        $medium->update([
            'name'       => $request->name,
            'slug'       => Str::slug($request->name),
            'form_id'    => $request->form_id,
            'status'     => $request->status ?? $medium->status,
            'updated_by' => Auth::guard('admin')->user()->name ?? 'admin',
        ]);

        return redirect()->back()->with('success', 'Medium updated successfully');
    }

    /**
     * DELETE
     */
    public function destroy($id)
    {
        $medium = Medium::withCount('paintings')->findOrFail($id);

        // Keep painting->medium links valid and avoid broken edit forms.
        if ($medium->paintings_count > 0) {
            return redirect()->back()
                ->with('error', 'Cannot delete medium. Paintings are linked to this medium.');
        }

        $medium->delete();

        return redirect()->back()->with('success', 'Medium deleted successfully');
    }
}
