<?php

namespace App\Http\Controllers\Admin;

use App\Models\Blog;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use App\Models\BlogCategory;

class BlogController extends Controller
{

   public function index()
{
    $blogs = Blog::latest()->paginate(10);

    return view('admin.blogs.index', compact('blogs'));
}

  public function store(Request $request)
{
    $request->validate([
        'blog_category_id'   => 'required|exists:blog_categories,id',
        'title'              => 'required|string|max:255',
        'slug'               => 'nullable|string|max:255',
        'short_description'  => 'required|string',
        'content'            => 'required|string',
        'status'             => 'required|in:draft,published',
        'image'              => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048',
        'seo_title'          => 'nullable|string|max:255',
        'meta_description'   => 'nullable|string|max:255',
    ]);

    // Auto-generate slug if empty
    $slug = $request->slug ?: Str::slug($request->title);

    // Handle image upload
    $imageName = null;
    if ($request->hasFile('image')) {
        $imageName = time() . '.' . $request->image->extension();
        $request->image->move(public_path('uploads/blogs'), $imageName);
    }

    Blog::create([
        'blog_category_id'  => $request->blog_category_id,
        'title'             => $request->title,
        'slug'              => $slug,
        'short_description' => $request->short_description,
        'content'           => $request->content,
        'status'            => $request->status,
        'image'             => $imageName,
        'seo_title'         => $request->seo_title,
        'meta_description'  => $request->meta_description,
    ]);

    return redirect()
        ->route('admin.blogs.index')
        ->with('success', 'Blog created successfully');
}

public function update(Request $request, Blog $blog)
{
    $data = $request->validate([
        'blog_category_id'   => 'required|exists:blog_categories,id',
        'title'              => 'required|string|max:255',
        'slug'               => 'nullable|string|max:255',
        'seo_title'          => 'nullable|string|max:255',
        'meta_description'   => 'nullable|string|max:500',
        'short_description'  => 'required|string|max:500',
        'status'             => 'required|in:draft,published',
        'content'            => 'required|string',
        'image'              => 'nullable|image|mimes:jpg,jpeg,png,webp|max:4096',
    ]);

    /* ---------- SLUG HANDLING ---------- */
    $slug = $data['slug'] ?: $data['title'];
    $slug = Str::slug($slug);

    if ($slug !== $blog->slug) {
        $originalSlug = $slug;
        $counter = 1;

        while (
            Blog::where('slug', $slug)
                ->where('id', '!=', $blog->id)
                ->exists()
        ) {
            $slug = $originalSlug . '-' . $counter++;
        }
    }

    $data['slug'] = $slug;

    /* ---------- IMAGE UPLOAD ---------- */
    if ($request->hasFile('image')) {
        if ($blog->image && file_exists(public_path('uploads/blogs/' . $blog->image))) {
            unlink(public_path('uploads/blogs/' . $blog->image));
        }

        $filename = time() . '_' . uniqid() . '.' . $request->image->getClientOriginalExtension();
        $request->image->move(public_path('uploads/blogs'), $filename);

        $data['image'] = $filename;
    }

    /* ---------- UPDATE ---------- */
    $blog->update($data);

    return redirect()
        ->route('admin.blogs.index')
        ->with('success', 'Blog updated successfully!');
}


public function create()
{
    $categories = BlogCategory::orderBy('name')->get();
    return view('admin.blogs.create', compact('categories'));
}

public function edit(Blog $blog)
{
    $categories = BlogCategory::orderBy('name')->get();

    return view('admin.blogs.edit', compact('blog', 'categories'));
}


public function destroy(Blog $blog)
{
    if ($blog->image && file_exists(public_path('uploads/blogs/'.$blog->image))) {
        @unlink(public_path('uploads/blogs/'.$blog->image));
    }

    $blog->delete();
    return back()->with('success', 'Blog deleted successfully!');
}

public function toggleStatus(Blog $blog)
{
    $blog->status = $blog->status === 'published' ? 'draft' : 'published';
    $blog->save();

    return back()->with('success', 'Blog status updated!');
}



}
