<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ArtworkType;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class ArtworkTypeController extends Controller
{
    public function index()
    {
        $artworks = ArtworkType::latest()->get();
        return view('admin.artworks.index', compact('artworks'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name'   => 'required|string|max:255|unique:artwork_types,name',
            'image'  => 'nullable|image|max:5120',
            'status' => 'required|in:0,1',
        ]);

        if ($request->hasFile('image')) {
            $filename = time().'_'.Str::random(6).'.'.$request->image->extension();
            $request->image->move(public_path('uploads/artworks'), $filename);
            $data['image'] = $filename;
        }

        $data['slug'] = Str::slug($data['name']);

        ArtworkType::create($data);

        return back()->with('success', 'Artwork type created.');
    }

    public function update(Request $request, $id)
    {
        $artwork = ArtworkType::findOrFail($id);

        $data = $request->validate([
            'name'   => 'required|string|max:255|unique:artwork_types,name,' . $artwork->id,
            'image'  => 'nullable|image|max:5120',
            'status' => 'required|in:0,1',
        ]);

        if ($request->hasFile('image')) {
            if ($artwork->image && file_exists(public_path('uploads/artworks/'.$artwork->image))) {
                @unlink(public_path('uploads/artworks/'.$artwork->image));
            }

            $filename = time().'_'.Str::random(6).'.'.$request->image->extension();
            $request->image->move(public_path('uploads/artworks'), $filename);
            $data['image'] = $filename;
        }

        // regenerate slug safely
        $data['slug'] = Str::slug($data['name']);

        $artwork->update($data);

        return back()->with('success', 'Artwork type updated.');
    }

    public function destroy($id)
    {
        $artwork = ArtworkType::findOrFail($id);

        if ($artwork->image && file_exists(public_path('uploads/artworks/'.$artwork->image))) {
            @unlink(public_path('uploads/artworks/'.$artwork->image));
        }

        $artwork->delete();

        return back()->with('success', 'Artwork type deleted.');
    }
}
